/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#ifndef lx_h
#define lx_h

#ifdef	__cplusplus
extern "C"
{
#endif


  /* Some of these may be overridden in mi.h */
#define LX_VERSION 2
#define LX_MAX_HOSTS 3000
#define LX_MAX_XBARS 1200
#define LX_SMALL_MTU 64
#define LX_BIG_MTU 1000
#define LX_ROUTE_SIZE 31
#define LX_EFFECTIVE_ROUTE_SIZE ((LX_ROUTE_SIZE - 1) / 2)
#define LX_XBAR_SIZE 32
#define LX_XBAR16_SIZE 16
#define LX_HOST_SIZE 2
#define LX_MAX_TIMEOUT_FACTOR 4
#define LX_TIMEOUT_INCREASE_FACTOR 2
#define LX_PASSIVE_TIMEOUT (1 * 1000 * 1000)
#define LX_ACTIVE_TIMEOUT (4 * 1000) /* overridden by MX */
#define LX_VERIFY_DISCONNECTED_TIMEOUT (1 * 1000 * 100)
#define LX_VERIFY_DISCONNECTED_TRIES 3
#define LX_VERIFY_CONNECTED_TIMEOUT (2 * 1000 * 1000)
#define LX_VERIFY_CONNECTED_TRIES 30
#define LX_BRANCH 2
#define LX_MAX_DEPTH 20
#define LX_NUM_SIMPLE_PASSES 8
#define LX_REPORT_TRIES 5
#define LX_PASSIVE_TRIES 3
#define LX_ACTIVE_TRIES 3
#define LX_COLLECT_TRIES 30
#define LX_COLLECT_TIMEOUT (1 * 1000 * 1000)
#define LX_TIMEOUT_DECREASE 500
#define LX_NO_PORT -LX_XBAR_SIZE
#define LX_SEEK_SIZE 1

enum
{
  LX_SMALL_SEND = 0,
  LX_REPLY_SEND,
  LX_BIG_SEND,
  LX_NUM_SEND_TYPES
};

enum
{
  LX_GM_HOST_TYPE = 1,
  LX_XM_HOST_TYPE = 2,
  LX_MX_HOST_TYPE = 3,
  LX_DEBUG_TYPE = 4,
  LX_TOOL_HOST_TYPE = 5,
  LX_NUM_HOST_TYPES = 5
};

enum
{
  LX_UP = 1,
  LX_DOWN = 2
};

enum
{
  LX_CHANGED_HOST = 0,
  LX_MISSING_HOST,
  LX_MISSING_LINK,
  LX_NEW_LINK,
  LX_CHANGED_ORDER_IN_COLLECT,
  LX_CHANGED_PORT_IN_COLLECT,
  LX_FAILED_ROUTING,
  LX_SCOUT_NOT_IN_MAP,
  LX_REQUEST_NOT_IN_MAP,
  LX_SCOUT_BAD_PARENT
};

/*
 * Flags for scout packets
 */
enum
{
  LX_FLAG_MAP_VALID = 0x1,
  LX_FLAG_KIDS_MATCH = 0x2,
  LX_FLAG_DONE_EVERYWHERE = 0x4,
  LX_FLAG_PAUSE = 0x8,
  LX_FLAG_MCP_REPLIED = 0x10,
  LX_FLAG_SUPPORTS_XBAR_32 = 0x20
};

enum
{
  LX_FLAG_ROUTING_CUSTOM,
  LX_FLAG_ROUTING_SPINE,
  LX_FLAG_ROUTING_SPINE_WITH_SHORTCUTS,
  LX_FLAG_ROUTING_SHORTEST_PATH,
  LX_FLAG_ROUTING_UP_DOWN,
  LX_FLAG_ROUTING_RANDOM,
  LX_FLAG_ROUTING_CAREFUL,
  LX_FLAG_ROUTING_VERY_CAREFUL,
  LX_FLAG_ROUTING_CAREFUL_SHORTCUTS
};

enum
{
  LX_FLAG_XBAR_VERIFY_NONE,
  LX_FLAG_XBAR_VERIFY_CLOS,
  LX_FLAG_XBAR_VERIFY_UP_DOWN,
  LX_FLAG_XBAR_VERIFY_ALL
};


enum
{
  LX_NONE,
  LX_RECEIVE,
  LX_ALARM,
  LX_SEND_DONE,
  LX_PUNT,
  LX_ABORT
};

#define LX_TYPE 0x700f
#define LX_SUBTYPE 0xa

enum
{
  LX_SCOUT = LX_SUBTYPE,
  LX_SCOUT_REPLY,
  LX_XBAR,
  LX_MAP_REQUEST,
  LX_MAP_REPLY,
  LX_CHANGE_REPORT,
  LX_XBAR32,
  LX_BIG_CHANGE_REPORT
};

enum
{
  LX_SCOUTING,
  LX_QUERYING_MAP_VERSION,
  LX_VERIFYING,
  LX_TREE,
  LX_CHANGE_REQUEST
};

#define lx_piece_count(t) (int)(((LX_BIG_MTU - sizeof (lx_map_reply_message_t)) / sizeof (t)))

enum
{
  LX_MAP_HOST = 0,
  LX_MAP_XBAR = 1,
  LX_MAP_XBAR32 = 3
};

#define LX_NUM_SMALLS (LX_XBAR_SIZE) /*don't need to send to oob ports*/
#define LX_NUM_REPLIES (LX_XBAR_SIZE/2) /*evil kludge to get under GM_NUM_SEND_TOKENS*/
#define LX_NUM_BIGS 4
#define LX_NUM_SENDS (LX_NUM_SMALLS + LX_NUM_REPLIES + LX_NUM_BIGS)

#define LX_MAX_LINE 100

#define lx_mac_format "%2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x"
#define lx_mac_args(m) m[0] & 0xff , m[1] & 0xff , m[2] & 0xff , m[3] & 0xff , m[4] & 0xff , m[5] & 0xff
#define lx_node_args(n) ((n)->xbar ? "x" : "h"), (n)->index
#define lx_node_format "%s%d"

#define lx_host_c(n)((lx_host_t*)(n))
#define lx_xbar_c(n)((lx_xbar_t*)(n))
#define lx_node_c(n)((lx_node_t*)(n))

struct lx_node_t;
struct lx_port_t;
struct lx_queue_t;

#ifdef	__cplusplus
extern "C"
{
#endif

#include "mi.h"

#ifdef	__cplusplus
}
#endif

typedef struct lx_route_t
{
  unsigned char length; /*must be 1st*/
  signed char hops [(LX_ROUTE_SIZE - 1) / 2];
}lx_route_t;

typedef struct lx_extended_t
{
  lx_route_t _r;
  signed char hops [LX_ROUTE_SIZE / 2 + 1];
}lx_extended_t;

struct lx_t;
struct lx_node_t;

typedef struct lx_event_t
{
  int type;
  char buffer [LX_SMALL_MTU];
}lx_event_t;

typedef struct lx_queue_t
{
  int count;
  struct lx_node_t*head;
  struct lx_node_t*tail;
}lx_queue_t;


typedef struct mi_link_t
{
  mi_index_t index;
  signed char port;
  mi_weight_t weight;
}mi_link_t;


typedef struct lx_node_t
{
  lx_queue_t*q;
  mi_index_t next;
  mi_index_t prev;
  char xbar;
  signed char first;
  
  mi_link_t original;
  unsigned short index;
  
  mi_node_data_t node_data;
  
  mi_link_t links [1];
}lx_node_t;

typedef struct lx_host_t
{
  lx_node_t _n;
  mi_link_t _l [LX_HOST_SIZE - 1];
  lx_route_t routes [MI_MY_SIZE] [LX_HOST_SIZE];
  unsigned char mac_address [6];
  unsigned char host_type;
  unsigned char level;
  unsigned char num_routes [LX_HOST_SIZE];
}lx_host_t;

typedef struct lx_xbar_t
{
  lx_node_t _n;
  mi_link_t _l [LX_XBAR_SIZE - 1];
  lx_route_t routes [MI_MY_SIZE];
  unsigned char level;
  unsigned char last;
  int id;
  int mask;
  unsigned char absolute_port;
  unsigned char up_skip_depth;	/* skip if going up and depth >= this */
  unsigned char down_skip_depth; /* skip if going down and depth >= this */
}lx_xbar_t;

typedef struct lx_header_t
{
  mi_u16_t type;
  mi_u16_t subtype;
  mi_u32_t timestamp;
  mi_u8_t sender_address [6];
  mi_u8_t echo_address [6];
  mi_s8_t port;
  mi_u8_t reason;
  mi_u8_t level;
  mi_u8_t version;
  mi_u32_t context;
}lx_header_t;


typedef struct lx_scout_message_t
{
  lx_header_t header;
  mi_u32_t map_version;
  mi_u8_t map_address [6];
  mi_u16_t flags;
  mi_u8_t route_length;
  signed char route [LX_EFFECTIVE_ROUTE_SIZE];

}lx_scout_message_t;

typedef struct lx_scout_reply_message_t
{
  lx_header_t header;
  mi_u32_t map_version;
  mi_u8_t map_address [6];
  mi_u8_t host_type;
  mi_u8_t iport;
  mi_u32_t flags;
}lx_scout_reply_message_t;

typedef struct lx_xbar_message_t
{
  lx_header_t header;
}lx_xbar_message_t;

/*these fields are constrained*/
/*by the hardware*/
typedef struct lx_xbar32_insert_t
{
  mi_u32_t id;              
  mi_u8_t absolute_port;     
  mi_u8_t quadrant_enabled;
  mi_u16_t pad0;
  mi_u32_t status_mask;
  mi_u32_t pad1;
  mi_u32_t crc32;
}lx_xbar32_insert_t;

typedef struct lx_xbar32_message_t
{
  mi_u16_t type;
  mi_u16_t subtype;
  lx_xbar32_insert_t insert;
  mi_u32_t timestamp;
  mi_u32_t context;
  mi_u8_t sender_address [6];
  mi_s8_t port;
  mi_u8_t version;
}lx_xbar32_message_t;

typedef struct lx_map_link_t
{
  mi_u16_t index;
  mi_s8_t port;
  mi_u8_t pad;
}lx_map_link_t;

typedef struct lx_map_host_t
{
  unsigned char mac_address [6];
  mi_u8_t host_type;
  mi_u8_t level;
  lx_map_link_t links [LX_HOST_SIZE];
}lx_map_host_t;

typedef struct lx_map_xbar_t
{
  mi_u8_t level;
  mi_u8_t pad [3];
  lx_map_link_t links [LX_XBAR_SIZE];
  mi_u16_t id_low;
  mi_u16_t id_high;  /*for padding bug in prev mappers*/
}lx_map_xbar_t;

typedef struct lx_map_xbar16_t
{
  mi_u8_t level;
  mi_u8_t pad [3];
  lx_map_link_t links [LX_XBAR16_SIZE];
}lx_map_xbar16_t;

typedef struct lx_map_request_message_t
{
  lx_header_t header;
  mi_u8_t type;
  mi_u8_t pad;
  mi_u16_t first;
  mi_u8_t route_length;
  signed char route [LX_EFFECTIVE_ROUTE_SIZE];
}lx_map_request_message_t;

typedef struct lx_map_reply_message_t
{
  lx_header_t header;
  mi_u8_t type;
  mi_u8_t routing_flag;
  mi_u16_t num_hosts;
  mi_u16_t num_xbars;
  mi_u16_t num_nodes;
  mi_u16_t first;
  unsigned char map_address [6];
  mi_u32_t map_version;
  mi_u8_t xbar_verify_flag;
  mi_u8_t pad;

  union
  {
    lx_map_host_t host [1];
    lx_map_xbar_t xbar [1];
    lx_map_xbar16_t xbar16 [1];
  } nodes;
  
}lx_map_reply_message_t;

typedef struct lx_old_map_reply_message_t
{
  lx_header_t header;
  mi_u8_t type;
  mi_u8_t routing_flag;
  mi_u16_t num_hosts;
  mi_u16_t num_xbars;
  mi_u16_t num_nodes;
  mi_u16_t first;
  unsigned char map_address [6];
  mi_u32_t map_version;
  mi_u8_t xbar_verify_flag;
  mi_u8_t pad;

  union
  {
    lx_map_host_t host [1];
    lx_map_xbar16_t xbar16 [1];
  } nodes;
  
}lx_old_map_reply_message_t;

typedef struct lx_change_t
{
  unsigned map_version;
  unsigned char map_address [6];
  unsigned char mac_address [6];
  unsigned char leaf_address [6];
  unsigned char child_address [6];
  int node;
  int port;
  int change;
  int pending;
  unsigned timestamp;
  mi_u32_t xid;
}lx_change_t;

typedef struct lx_change_report_message_t
{
  lx_header_t header;
  mi_u32_t map_version;
  mi_u8_t map_address [6];
  mi_u8_t mac_address [6];
  mi_u16_t node;
  mi_u8_t port;
  mi_u8_t change;
  mi_u8_t route_length;
  signed char route [LX_EFFECTIVE_ROUTE_SIZE];
}lx_change_report_message_t;

typedef struct lx_big_change_report_message_t
{
  lx_change_report_message_t change;
  mi_u32_t xid;
}lx_big_change_report_message_t;

typedef union lx_packet_t
{
  lx_scout_message_t scout;
  lx_scout_reply_message_t scout_reply;
  lx_xbar_message_t xbar;
  lx_map_request_message_t map_request;
  lx_map_reply_message_t map_reply;
  lx_change_report_message_t change_report;
  lx_big_change_report_message_t big_change_report;
}lx_packet_t;

typedef struct lx_map_t
{
  struct lx_t*lx;
  lx_queue_t xbars;
  lx_queue_t hosts;
  lx_queue_t frontier;
  lx_queue_t merge_queue;
  mi_index_t host_array [LX_MAX_HOSTS];
  mi_index_t xbar_array [LX_MAX_XBARS];

  lx_node_t*root;
  int num_xbars;
  int num_hosts;
  unsigned map_valid;
  unsigned routeable;
  unsigned active;
  unsigned mapping;
  unsigned routing;
  unsigned char map_address [6];
  unsigned char winner_address [6];
  lx_route_t winner_route;  
  int winner_iport;
  unsigned map_version;
  unsigned map_counter;
  int done_everywhere;
  unsigned char parent_address [6];
  char pad [2];
  lx_route_t parent_route;
  int parent_iport;
  unsigned kids_match;
  unsigned max_level;
  unsigned routing_flag;
  unsigned xbar_verify_flag;
  unsigned seek_size;
  unsigned has_x32s;
}lx_map_t;

#ifdef __GNUC__
#define lx_aligned __attribute__ ((aligned(8)))
#else
#define lx_aligned
#endif

typedef struct lx_buffer_t
{
  int type;
  struct lx_buffer_t*next;
} lx_buffer_t;

typedef struct lx_small_t
{
  lx_buffer_t _b;
  lx_route_t r;
  lx_route_t _r;
  char p [LX_SMALL_MTU];
}lx_small_t lx_aligned;


typedef struct lx_big_t
{
  lx_buffer_t _b;
  lx_route_t r;
  lx_route_t _r;
  char p [LX_BIG_MTU];
}lx_big_t lx_aligned;

typedef struct lx_t
{
  mi_t mi;
  
  unsigned int aborted;
  unsigned int die;
  lx_map_t maps [MI_MY_SIZE];
  lx_change_t changes [MI_MY_SIZE];

  lx_host_t host_block [LX_MAX_HOSTS];
  lx_xbar_t xbar_block [LX_MAX_XBARS];
  lx_queue_t free_hosts;
  lx_queue_t free_xbars;

  lx_queue_t staged_hosts;
  lx_queue_t staged_xbars;  

  lx_small_t reply_block [LX_NUM_REPLIES];
  lx_small_t small_block [LX_NUM_SMALLS];
  lx_big_t big_block [LX_NUM_BIGS];

  lx_buffer_t*sends [LX_NUM_SEND_TYPES];
  unsigned pendings [LX_NUM_SEND_TYPES];
  unsigned maxes  [LX_NUM_SEND_TYPES];
  
  char receive_buffer [LX_BIG_MTU];
  char map_reply_buffer [LX_BIG_MTU];
  
  unsigned replies_sent;
  unsigned smalls_sent;
  unsigned bigs_sent;
  
  unsigned duration;
  unsigned char set_alarm;
  
  unsigned char mac_address [6];
  unsigned char pause_address [6];
  
  unsigned timestamp;
  unsigned last_seen_timestamp;
  unsigned start_timeout;
  unsigned timeout;
  unsigned max_level;
  int tries;
  int extra_hosts;
  int num_ports;
  int thrash;

  unsigned level;
  unsigned host_type;
  unsigned map_once;
  unsigned pause;
  unsigned paused;
  unsigned net_paused;
  unsigned routing_flag;
  char *route_filebase;
  unsigned xbar_verify_flag;
  unsigned seek_size;
  int num_routes;
  int active_tries;
  
  unsigned in_receive;
  unsigned first_host;
  int first_host_type;
  lx_map_t*first_host_map;
}lx_t;

void lx_init (lx_t*lx);
void lx_reset_map (lx_t*lx, lx_map_t*m);

lx_node_t*lx_new_xbar (lx_map_t*m, int iport, lx_route_t*route, int id, int mask, int absolute_port);
lx_node_t*lx_new_host (lx_map_t*m, int iport, int oiport, unsigned char mac_address [6], int host_type, int level, lx_route_t*route);

void lx_delete_node (lx_map_t*m, lx_node_t*node);
void lx_delete_hosts (lx_map_t*m, lx_node_t*n);
int lx_connect (lx_node_t*na, int pa, lx_node_t*nb, int pb);
int lx_map_connect (lx_node_t*na, int pa, int index, int pb);
void lx_disconnect (lx_node_t*n, int p);
lx_node_t*lx_get_node (lx_node_t*n, int port);
#define lx_get_port(n, i)((n)->links[i - n->first].port)
#define lx_connected(n, p)(lx_get_node (n, p) ? 1 : 0)

int lx_get_xbar_count (lx_node_t*n);
int lx_oob (lx_node_t*n, int port);
int lx_there_and_back_again (signed char*dst, signed char*src, signed char port, int length);
void lx_copy_route (signed char*dst, signed char*src, int length);
void lx_reverse_route (signed char*dst, signed char*src, int length);
void lx_physical_route (signed char*p, int length);
void lx_logical_route (signed char*p, int length);
lx_route_t*lx_append_route (lx_route_t*dst, lx_route_t*route, signed char port);
void lx_compose_route (lx_route_t*dst, lx_route_t*a, signed char hop, lx_route_t*b);

char*lx_print_route (signed char*src, int length);
char*lx_print_change (int change);

int lx_put (lx_queue_t*q, lx_node_t*n);
int lx_put_at_front (lx_queue_t*q, lx_node_t*n);
int lx_insert (lx_queue_t*q, lx_node_t*before, lx_node_t*n);
lx_node_t*lx_get (lx_queue_t*q);
lx_node_t*lx_peek (lx_queue_t*q);
int lx_queued (lx_queue_t*q, lx_node_t*n);
lx_node_t*lx_remove (lx_queue_t*q, lx_node_t*n);
lx_node_t*lx_remove_from_anywhere (lx_node_t*n);
lx_node_t*lx_seen (lx_map_t*m, unsigned char mac_address [6]);

int lx_map_print (lx_t*lx, lx_map_t*m);
int lx_firsts (lx_queue_t*queue);
unsigned lx_dice (unsigned throws, unsigned sides);
int lx_memcmp (unsigned char*a, unsigned char*b, unsigned length);
int lx_zero (unsigned char*p, unsigned length);
int lx_has_hosts (lx_node_t*n);

int lx_number_xbars_bfs (lx_map_t*m, int iport);
int lx_before_sending (lx_map_t*m);
void lx_before_receiving (lx_map_t*m, int num_hosts, int num_xbars);
void lx_bzero (char*p, unsigned length);
void lx_set_timeout (lx_t*lx, unsigned timeout, unsigned tries);

lx_node_t*lx_follow (lx_node_t*n, int p, lx_route_t*r, int*last_port);
int lx_higher_than_me (lx_t*lx, unsigned char mac_address [6], int level);
int lx_hostcmp (unsigned char address1 [6], int level1, unsigned char address2 [6], int level2);

void lx_reclaim_staged (lx_t*lx);
lx_buffer_t*lx_alloc_send (lx_t*lx, int type);
void lx_free_send (lx_t*lx, lx_buffer_t*p);
lx_node_t*lx_find_host (lx_t*lx, lx_queue_t*queue, unsigned char mac_address [6], int level, lx_node_t**higher);
lx_node_t*lx_find_xbar (lx_t*lx, lx_queue_t*queue, int id);

int lx_inflate_map (lx_map_t*m, int iport, int xbar_fan, int extra_hosts);
int lx_num_debug_hosts (lx_map_t*m);
int lx_get_highest_mac_address (lx_map_t*m, unsigned char mac_address [6]);
lx_node_t*lx_bsearch_host (lx_map_t*m, unsigned char mac_address [6], int level);

int lx_compare_time (lx_t*lx, unsigned received, unsigned expected);
unsigned lx_advance_time (lx_t*lx);

#define LXD_MAX_LINES 64
extern int lxd_cur;
extern int lxd_lines [LXD_MAX_LINES];
void lxd_line (int line);

void lx_dump (char*p, int length);
#define lx_iport(m)(m - m->lx->maps)
void lx_mapper_main (lx_t*lx);
int lx_wait (lx_t*lx, unsigned duration, int*variable, int early);

unsigned char*lx_get_next_host (lx_t*lx);
unsigned char*lx_get_first_host (lx_t*lx, int iport, int host_type);

void lx_seed (unsigned s);
unsigned lx_rand (void);
int lx_set_verbosity(int state);
int lx_set_change (lx_t*lx, int iport, unsigned char child_address [6], unsigned timestamp, int change, unsigned map_version, unsigned char map_address [6], unsigned char leaf_address [6], unsigned char mac_address [6], int node, int port, unsigned xid);
int lx_0011 (lx_t*lx, int n);


int lx_in_bounds (lx_node_t*n, int port);

static inline int lx_byte_swap_long (int n)
{
  n = mi_htonl (n);
  return (((n>>24)&0xff) | ((n>>8)&0xff00) | ((n<<8)&0xff0000) | ((n<<24)&0xff000000));

}

#ifdef __cplusplus
#if 0
{
#endif
}
#endif

#endif /*lx_h*/
